// Content script to scan web pages for Telegram bot tokens

// Telegram bot token regex patterns - Enhanced ID:HASH format detection
const TOKEN_PATTERNS = [
  // Primary ID:HASH format patterns (8-10 digits:35 chars)
  /\b(\d{8,10}:[A-Za-z0-9_-]{35})\b/g,
  /\b(\d{9,10}:[A-Za-z0-9_-]{35})\b/g,
  /\b(\d{10}:[A-Za-z0-9_-]{35})\b/g,
  
  // Bot token in URLs (api.telegram.org/botTOKEN/...)
  /\/bot(\d{8,10}:[A-Za-z0-9_-]{35})/g,
  // Bot token in URLs without /bot prefix
  /(\d{8,10}:[A-Za-z0-9_-]{35})/g,
  
  // Enhanced variable patterns
  /bot[_-]?token["\s]*[:=]["\s]*([A-Za-z0-9_-]{40,})/gi,
  /telegram[_-]?bot[_-]?token["\s]*[:=]["\s]*([A-Za-z0-9_-]{40,})/gi,
  /tg[_-]?bot[_-]?token["\s]*[:=]["\s]*([A-Za-z0-9_-]{40,})/gi,
  /telegram[_-]?api[_-]?key["\s]*[:=]["\s]*([A-Za-z0-9_-]{40,})/gi,
  /bot[_-]?api[_-]?key["\s]*[:=]["\s]*([A-Za-z0-9_-]{40,})/gi,
  
  // API key patterns
  /api[_-]?key["\s]*[:=]["\s]*([A-Za-z0-9_-]{40,})/gi,
  
  // Token in JavaScript variables and strings (ID:HASH format)
  /['""]([0-9]{8,10}:[A-Za-z0-9_-]{35})['""]/g,
  /['""]([A-Za-z0-9_-]{40,})['""]/g,
  
  // Token in configuration files (JSON, YAML, etc.)
  /["']([0-9]{8,10}:[A-Za-z0-9_-]{35})["']/g,
  /["']([A-Za-z0-9_-]{40,})["']/g,
  
  // Token in environment variables
  /[A-Z_]+[_-]?TOKEN[_-]?[A-Z_]*["\s]*[:=]["\s]*([A-Za-z0-9_-]{40,})/gi,
  /[A-Z_]+[_-]?API[_-]?KEY[_-]?[A-Z_]*["\s]*[:=]["\s]*([A-Za-z0-9_-]{40,})/gi,
  
  // URL query parameters and fragments
  /[?&]token=([A-Za-z0-9_-]{40,})/gi,
  /[?&]api[_-]?key=([A-Za-z0-9_-]{40,})/gi,
  /[?&]bot[_-]?token=([A-Za-z0-9_-]{40,})/gi,
  
  // Base64 encoded or obfuscated patterns
  /[A-Za-z0-9+/]{50,}={0,2}/g
];

// Function to validate ID:HASH format token
function isValidTelegramToken(token) {
  // Primary validation: ID:HASH format (8-10 digits:35 characters)
  const primaryPattern = /^\d{8,10}:[A-Za-z0-9_-]{35}$/;
  if (primaryPattern.test(token)) {
    return true;
  }
  
  // Secondary validation: Longer tokens that might be valid
  const secondaryPattern = /^\d{8,10}:[A-Za-z0-9_-]{35,}$/;
  if (secondaryPattern.test(token)) {
    return true;
  }
  
  return false;
}

// Function to extract tokens from text
function extractTokens(text) {
  const tokens = new Set();
  
  TOKEN_PATTERNS.forEach(pattern => {
    let match;
    while ((match = pattern.exec(text)) !== null) {
      // Extract the token from the match
      let token = match[1] || match[0];
      
      // Clean up the token
      token = token.replace(/['""]/g, '').trim();
      
      // Additional cleaning for URL fragments and query parameters
      token = token.replace(/[\/\?&].*$/, ''); // Remove everything after /, ?, or &
      token = token.replace(/^.*[=:]/, ''); // Remove key= or key: prefix
      
      // Skip if token is too short or doesn't contain a colon
      if (token.length < 40 || !token.includes(':')) {
        continue;
      }
      
      // Validate token format using enhanced validation
      if (isValidTelegramToken(token)) {
        tokens.add(token);
      }
    }
  });
  
  // Enhanced URL scanning for ID:HASH format
  const urlPatterns = [
    /[?&\/]bot(\d{8,10}:[A-Za-z0-9_-]{35,})/g,
    /[?&]token=(\d{8,10}:[A-Za-z0-9_-]{35,})/gi,
    /[?&]api[_-]?key=(\d{8,10}:[A-Za-z0-9_-]{35,})/gi,
    /[?&]bot[_-]?token=(\d{8,10}:[A-Za-z0-9_-]{35,})/gi
  ];
  
  urlPatterns.forEach(pattern => {
    let match;
    while ((match = pattern.exec(text)) !== null) {
      const token = match[1];
      if (isValidTelegramToken(token)) {
        tokens.add(token);
      }
    }
  });
  
  return Array.from(tokens);
}

// Function to scan page content
function scanPage() {
  try {
    // Check if extension context is still valid before scanning
    if (!isExtensionContextValid()) {
      console.debug('Extension context invalidated, skipping page scan');
      return;
    }
    
    const pageText = document.documentElement.innerHTML;
    const pageTextContent = document.body ? document.body.innerText : '';
    const currentUrl = window.location.href;
    
    // Extract tokens from HTML content
    const htmlTokens = extractTokens(pageText);
    
    // Extract tokens from text content
    const textTokens = extractTokens(pageTextContent);
    
    // Extract tokens from current URL
    const urlTokens = extractTokens(currentUrl);
    
    // Combine and deduplicate tokens
    const allTokens = [...new Set([...htmlTokens, ...textTokens, ...urlTokens])];
    
    if (allTokens.length > 0) {
      console.log(`Found ${allTokens.length} potential Telegram bot tokens:`, allTokens);
      console.log('Current URL:', currentUrl);
      
      // Send tokens to background script for validation
      safeSendMessage({
        type: 'TOKENS_FOUND',
        tokens: allTokens,
        url: window.location.href,
        title: document.title,
        timestamp: Date.now()
      });
    }
  } catch (error) {
    if (error.message.includes('Extension context invalidated')) {
      console.debug('Extension context invalidated during page scan');
      cleanup();
    } else {
      console.error('Error during page scan:', error);
    }
  }
}

// Function to scan network responses
function interceptNetworkRequests() {
  // Override fetch to intercept responses
  const originalFetch = window.fetch;
  window.fetch = function(...args) {
    return originalFetch.apply(this, args).then(response => {
      // Clone the response to read it without consuming it
      const clonedResponse = response.clone();
      
      // Check if response contains text
      const contentType = response.headers.get('content-type');
      if (contentType && contentType.includes('text')) {
        clonedResponse.text().then(text => {
          if (!isExtensionContextValid()) {
            console.debug('Extension context invalidated, skipping network response scan');
            return;
          }
          
          const tokens = extractTokens(text);
          if (tokens.length > 0) {
            console.log(`Found tokens in network response:`, tokens);
            safeSendMessage({
              type: 'TOKENS_FOUND',
              tokens: tokens,
              url: response.url,
              title: `Network Response: ${response.url}`,
              timestamp: Date.now()
            });
          }
        }).catch(error => {
          // Ignore errors when reading response text
          console.debug('Could not read response text:', error);
        });
      }
      
      return response;
    });
  };
  
  // Override XMLHttpRequest to intercept responses
  const originalXHROpen = XMLHttpRequest.prototype.open;
  const originalXHRSend = XMLHttpRequest.prototype.send;
  
  XMLHttpRequest.prototype.open = function(method, url, ...args) {
    this._url = url;
    return originalXHROpen.apply(this, [method, url, ...args]);
  };
  
  XMLHttpRequest.prototype.send = function(...args) {
    const xhr = this;
    const originalOnReadyStateChange = xhr.onreadystatechange;
    
    xhr.onreadystatechange = function() {
      if (xhr.readyState === 4 && xhr.status === 200) {
        const contentType = xhr.getResponseHeader('content-type');
        if (contentType && contentType.includes('text')) {
          if (!isExtensionContextValid()) {
            console.debug('Extension context invalidated, skipping XHR response scan');
            return;
          }
          
          const tokens = extractTokens(xhr.responseText);
          if (tokens.length > 0) {
            console.log(`Found tokens in XHR response:`, tokens);
            safeSendMessage({
              type: 'TOKENS_FOUND',
              tokens: tokens,
              url: xhr._url || 'XHR Request',
              title: `XHR Response: ${xhr._url}`,
              timestamp: Date.now()
            });
          }
        }
      }
      
      if (originalOnReadyStateChange) {
        originalOnReadyStateChange.apply(this, arguments);
      }
    };
    
    return originalXHRSend.apply(this, args);
  };
}

// Background scanning settings
let scanningSettings = {
  enabled: true,
  interval: 5000, // 5 seconds
  scanOnNavigation: true,
  scanNetworkRequests: true,
  scanPageContent: true,
  scanURL: true,
  autoValidate: true
};

// Background scanning interval
let scanningInterval = null;

// Check if extension context is still valid
function isExtensionContextValid() {
  try {
    return chrome.runtime && chrome.runtime.id;
  } catch (error) {
    return false;
  }
}

// Safe message sending with context validation
async function safeSendMessage(message) {
  if (!isExtensionContextValid()) {
    console.debug('Extension context invalidated, skipping message:', message.type);
    return null;
  }
  
  try {
    return await chrome.runtime.sendMessage(message);
  } catch (error) {
    if (error.message.includes('Extension context invalidated')) {
      console.debug('Extension context invalidated during message send');
      return null;
    }
    throw error;
  }
}

// Load settings from background script
async function loadSettings() {
  try {
    const response = await safeSendMessage({ type: 'GET_SETTINGS' });
    if (response && response.settings) {
      scanningSettings = { ...scanningSettings, ...response.settings };
    }
  } catch (error) {
    console.debug('Could not load settings:', error);
  }
}

// Debug function to test token detection
function debugTokenDetection() {
  const testTexts = [
    "api.telegram.org/bot7963974508:AAGyCacGCKKxa2fnWawtieZGtfg2VtKX5ps/sendMessage?chat_id=7935143652&text=t31kk",
    "7963974508:AAGyCacGCKKxa2fnWawtieZGtfg2VtKX5ps",
    "const botToken = '7963974508:AAGyCacGCKKxa2fnWawtieZGtfg2VtKX5ps';",
    "BOT_TOKEN=7963974508:AAGyCacGCKKxa2fnWawtieZGtfg2VtKX5ps",
    "telegram_bot_token: 7963974508:AAGyCacGCKKxa2fnWawtieZGtfg2VtKX5ps"
  ];
  
  console.log("=== Telegram Bot Token Scanner Debug ===");
  console.log("Current scanning settings:", scanningSettings);
  
  testTexts.forEach((testText, index) => {
    const foundTokens = extractTokens(testText);
    console.log(`Test ${index + 1}: "${testText}"`);
    console.log(`Found tokens:`, foundTokens);
    console.log("---");
  });
  
  console.log("=== End Debug ===");
}

// Start background scanning
function startBackgroundScanning() {
  if (scanningInterval) {
    clearInterval(scanningInterval);
    scanningInterval = null;
  }
  
  if (scanningSettings.enabled && scanningSettings.interval > 0) {
    try {
      scanningInterval = setInterval(() => {
        try {
          if (!isExtensionContextValid()) {
            console.debug('Extension context invalidated during background scan');
            cleanup();
            return;
          }
          
          if (scanningSettings.scanPageContent) {
            scanPage();
          }
        } catch (error) {
          if (error.message.includes('Extension context invalidated')) {
            console.debug('Extension context invalidated during background scan');
            cleanup();
          } else {
            console.error('Error during background scan:', error);
          }
        }
      }, scanningSettings.interval);
      
      console.log(`Background scanning started with ${scanningSettings.interval}ms interval`);
    } catch (error) {
      console.error('Error starting background scanning:', error);
    }
  }
}

// Stop background scanning
function stopBackgroundScanning() {
  if (scanningInterval) {
    clearInterval(scanningInterval);
    scanningInterval = null;
    console.log('Background scanning stopped');
  }
}

// Initialize scanning
async function initializeScanner() {
  try {
    // Check if extension context is still valid
    if (!isExtensionContextValid()) {
      console.log('Extension context invalidated, skipping initialization');
      return;
    }
    
    // Load settings first
    await loadSettings();
    
    // Debug token detection
    debugTokenDetection();
    
    // Initial page scan
    if (scanningSettings.scanPageContent) {
      scanPage();
    }
    
    // Set up network request interception
    if (scanningSettings.scanNetworkRequests) {
      interceptNetworkRequests();
    }
    
    // Start background scanning
    startBackgroundScanning();
    
    // Set up message listener
    setupMessageListener();
    
    // Scan page again after a delay to catch dynamically loaded content
    setTimeout(() => {
      if (scanningSettings.scanPageContent && isExtensionContextValid()) {
        scanPage();
      }
    }, 2000);
    
    // Listen for page changes (for SPAs)
    if (scanningSettings.scanOnNavigation) {
      let lastUrl = location.href;
      let mutationObserver = null;
      
      try {
        mutationObserver = new MutationObserver(() => {
          try {
            if (!isExtensionContextValid()) {
              cleanup();
              return;
            }
            
            const url = location.href;
            if (url !== lastUrl) {
              lastUrl = url;
              setTimeout(() => {
                try {
                  if (scanningSettings.scanPageContent && isExtensionContextValid()) {
                    scanPage();
                  }
                } catch (error) {
                  if (error.message.includes('Extension context invalidated')) {
                    console.debug('Extension context invalidated during navigation scan');
                    cleanup();
                  } else {
                    console.error('Error during navigation scan:', error);
                  }
                }
              }, 1000);
            }
          } catch (error) {
            if (error.message.includes('Extension context invalidated')) {
              console.debug('Extension context invalidated during mutation observer');
              cleanup();
            } else {
              console.error('Error in mutation observer:', error);
            }
          }
        });
        
        mutationObserver.observe(document, { subtree: true, childList: true });
        console.log('Mutation observer started for navigation detection');
      } catch (error) {
        console.error('Error setting up mutation observer:', error);
      }
    }
  } catch (error) {
    if (error.message.includes('Extension context invalidated')) {
      cleanup();
    } else {
      console.error('Error during initialization:', error);
    }
  }
}

// Cleanup function for extension context invalidation
function cleanup() {
  stopBackgroundScanning();
  console.log('Extension context invalidated, cleanup completed');
}

// Listen for settings changes with error handling
function setupMessageListener() {
  try {
    chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
      if (!isExtensionContextValid()) {
        cleanup();
        return;
      }
      
      if (message.type === 'SETTINGS_UPDATED') {
        scanningSettings = { ...scanningSettings, ...message.settings };
        
        // Restart background scanning with new settings
        stopBackgroundScanning();
        startBackgroundScanning();
        
        console.log('Settings updated, scanning restarted:', scanningSettings);
      }
    });
  } catch (error) {
    if (error.message.includes('Extension context invalidated')) {
      cleanup();
    } else {
      console.error('Error setting up message listener:', error);
    }
  }
}

// Global error handler for extension context invalidation
window.addEventListener('error', (event) => {
  if (event.error && event.error.message && event.error.message.includes('Extension context invalidated')) {
    console.debug('Extension context invalidated - cleaning up');
    cleanup();
    event.preventDefault();
    return false;
  }
});

window.addEventListener('unhandledrejection', (event) => {
  if (event.reason && event.reason.message && event.reason.message.includes('Extension context invalidated')) {
    console.debug('Extension context invalidated - cleaning up');
    cleanup();
    event.preventDefault();
    return false;
  }
});

// Start scanning when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initializeScanner);
} else {
  initializeScanner();
}
